/**
 * \par Copyright (C), 2012-2016, MakeBlock
 * \class MeDCMotor
 * \brief   Driver for Me DC motor device.
 * @file    MeDCMotor.h
 * @author  MakeBlock
 * @version V1.0.1
 * @date    2016/04/07
 * @brief   Header for MeDCMotor.cpp module
 *
 * \par Copyright
 * This software is Copyright (C), 2012-2016, MakeBlock. Use is subject to license \n
 * conditions. The main licensing options available are GPL V2 or Commercial: \n
 *
 * \par Open Source Licensing GPL V2
 * This is the appropriate option if you want to share the source code of your \n
 * application with everyone you distribute it to, and you also want to give them \n
 * the right to share who uses it. If you wish to use this software under Open \n
 * Source Licensing, you must contribute all your source code to the open source \n
 * community in accordance with the GPL Version 2 when your application is \n
 * distributed. See http://www.gnu.org/copyleft/gpl.html
 *
 * \par Description
 * This file is a drive for Me DC motor device.
 *
 * \par Method List:
 *
 *    1. void MeDCMotor::setpin(uint8_t dir_pin,uint8_t pwm_pin)
 *    2. void MeDCMotor::run(int16_t speed)
 *    3. void MeDCMotor::stop(void)
 *    4. void MeDCMotor::reset(uint8_t port)
 *    5. void MeDCMotor::reset(uint8_t port, uint8_t slot)
 *
 * \par History:
 * <pre>
 * `<Author>`         `<Time>`        `<Version>`        `<Descr>`
 * Mark Yan         2015/09/09     1.0.0            Rebuild the old lib.
 * Mark Yan         2016/04/07     1.0.1            fix motor reset issue.
 * </pre>
 *
 */
#ifndef MeDCMotor_H
#define MeDCMotor_H

#include <stdint.h>
#include <stdbool.h>
#include <Arduino.h>
#include "MeConfig.h"

#include "MeGearAngle.h"


#ifdef ME_PORT_DEFINED
#include "MePort.h"
#endif


#define MIN_PWM_DCSPEED 20000

#define MAX_PWM_DCSPEED 100000

#define MIN_DC_SPEED 0
#define MAX_DC_SPEED 160

#define USER_MIN_DCSPEED (-130)
#define USER_MAX_DCSPEED (130)

#define USER_MIN_STOP_DCSPEED (-9)
#define USER_MAX_STOP_DCSPEED (9)



#define DCSPEED2PWD(a) ((((a)>MAX_DC_SPEED?MAX_DC_SPEED:(a)) * (MAX_PWM_DCSPEED-MIN_PWM_DCSPEED) / MAX_DC_SPEED) + MIN_PWM_DCSPEED )


#define MIN_CYCLE_TIME 1 //200 us
#define MAX_CYCLE_TIME 1000 // 1ms


#define MAX_BIT_DIV 100

#define LENTH_MOTOR_LT 204
#define GA_COUNT 20 //10//20

#define IQCAR_WIDTH 120
#define LT_WIDTH 26

#define IQCAR_CYCLE_LEN 592 //((IQCAR_WIDTH-LT_WIDTH)*2*314/100) //672 //842 //408

#define IQCAR_CYCLE_GEARS 56 //64

#define MIN_START_SPEED 80

#define MAX_RORATE_ANGLE 14400
#define DC_BRAKING_GEAR 20 
#define RUNWITHANGLE_TIMEOUT 5000
#define MIN_CAN_RUN_SPEED 10 //5 //10


//#define SOFTTIMER_PWM_CONTROL_MODE

#define HARDWARE_PWM_CONTROL_MODE


#ifdef HARDWARE_PWM_CONTROL_MODE

#define MAX_PWM_ANALOG 250

#define MIN_PWM_ANALOG 80

#endif

/**
 * Class: MeDCMotor
 * \par Description
 * Declaration of Class MeDCMotor
 */
class MeDCMotor
{
public:
  MeDCMotor(void);
/**
 * \par Function
 *   setpin
 * \par Description
 *   Reset the DC motor available PIN by its arduino port.
 * \param[in]
 *   dir_pin - arduino port for direction pin(should analog pin)
 * \param[in]
 *   pwm_pin - arduino port for pwm input(should analog pin)
 * \par Output
 *   None
 * \return
 *   None
 * \par Others
 *   None
 */
  void setpin(uint8_t fin_pin,uint8_t rin_pin, uint8_t vcc_pin, uint8_t angle_pin);

/**
 * \par Function
 *   reset
 * \par Description
 *   Reset the DC motor available PIN by its RJ25 port.
 * \param[in]
 *   port - RJ25 port from PORT_1 to M2
 * \par Output
 *   None
 * \return
 *   None
 * \par Others
 *   None
 */
  void reset(uint8_t port);

/**
 * \par Function
 *   reset
 * \par Description
 *   Reset the DC motor available PIN by its RJ25 port and slot.
 * \param[in]
 *   port - RJ25 port from PORT_1 to M2
 * \param[in]
 *   slot - SLOT1 or SLOT2
 * \par Output
 *   None
 * \return
 *   None
 * \par Others
 *   None
 */
  void reset(uint8_t port, uint8_t slot);

/**
 * \par Function
 *   run
 * \par Description
 *   Control the motor forward or reverse
 * \param[in]
 *   speed - Speed value from -255 to 255
 * \par Output
 *   None
 * \return
 *   None
 * \par Others
 *   None
 */
  void run(int16_t speed);

  void RunByPWM(uint32_t nRLPWM, uint32_t nFRPWM);

  void RunByGPIO(int16_t nMode);


/**
 * \par Function
 *   stop
 * \par Description
 *   Stop the rotation of the motor
 * \par Output
 *   None
 * \return
 *   None
 * \par Others
 *   None
 */
  void stop(void);

  void TurnWithAngle( int nSpeed, int Angles);

  long GetCode(void);

  void SetCode(void);

  long GetRawGear(void);

  void SetGSCallback(voidGSFuncPtr pCallback){meGA.SetGSCallback(pCallback);};
  voidGSFuncPtr GetGSCallback(void){return meGA.GetGSCallback();};

  

#ifdef SOFTTIMER_PWM_CONTROL_MODE
  void OnPWMTimer(void);

  void BeginPWM(uint32_t nCycle);

  void ResetPWM(void){nCycleCount = 0;};
 #endif 
 
private:
  volatile uint8_t dc_fin_pin;
  volatile uint8_t dc_rin_pin;
  volatile uint8_t dc_vcc_pin;
  volatile uint8_t dc_angle_pin;
  int16_t  last_speed;
  int16_t  last_dir;
  

#ifdef SOFTTIMER_PWM_CONTROL_MODE
  uint32_t nCycleTime;
  uint32_t nCycleCount;

  uint32_t nLeftPWMFreq;
  uint32_t nRightPWMFreq;

  uint32_t nLeftPlus;
  uint32_t nRightPlus;
#endif

  unsigned int nLastLeftAngle;
  unsigned int nLastRightAngle;

#ifdef SOFTTIMER_PWM_CONTROL_MODE
  bool bPauseRun;
  bool bTimerPause;
  unsigned int nDivSpeed;
  uint8_t bitdivmask[MAX_BIT_DIV+10];
#endif
  
  

  MeGearAngle meGA;
  
};


#ifdef __cplusplus
extern "C" {
#endif

extern void DC_SetMotorCode(uint8_t bRight);

extern long DC_GetMotorCode(uint8_t bRight);

extern long DC_GetMotorRawGear(uint8_t bRight);


extern void DC_TurnWithAngle(uint8_t bRight, int16_t nSpeed, int16_t nAngle);

extern void DC_Run(int8_t bRight, int16_t nSpeed );

extern void DC_RunWithAngle(uint8_t bRight, int16_t nSpeed, long nAngle);

extern void DC_SetGSCallback(uint8_t bRight, voidGSFuncPtr pCallback);

extern voidGSFuncPtr DC_GetGSCallback(uint8_t bRight);

extern void DC_RunByPWM(int8_t bRight, uint32_t nRLPWM,uint32_t nFRPWM );

extern void DC_RunByGPIO(uint8_t nRight, int16_t nMode );



#ifdef __cplusplus
}
#endif


#endif

